/**
  ******************************************************************************
  * @file    main.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   Program entry point
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

/* Includes ------------------------------------------------------------------*/
#include "fibonacci.h"
#include "stm32f4xx_hal.h"

volatile unsigned long g_MSCtr;

unsigned long g_SaveStack;

extern void FlashBurnFlash(char *cRomImage,int iRomSize, char *cWhere);
extern unsigned long _sidata;

unsigned long g_FibonacciResults[FIBTEST_USE_MAX][FIBTEST_MAX][MAXFIBONACCITESTS][FIBTEST_RESULTITEM_ALL];

FibonacciFn g_FibCallableFn;
FibonacciFn g_FibCallableFn0;

#define DWT_CTRL     0xE0001000
#define DWT_CYCCNT   0xE0001004
#define DEMCR        0xE000EDFC

/** @brief sets up the DWT to count CPU cycles. 
 *
 *
 *  @return none
 */

void SetupCortexDWTCycleCounter( void )
{
    // This sequence activates the DWT cycle counter. May interfere with DWT based breakpoints!
    (*(volatile unsigned int*)DEMCR) |= (1 << 24);
    (*(volatile unsigned int*)DWT_CYCCNT) = 0;
    (*(volatile unsigned int*)DWT_CTRL) |= (1);    
}

#define configCPU_CLOCK_HZ				( 168000000UL )	
#define configTICK_RATE_HZ				( 1000 )
#define configSYSTICK_CLOCK_HZ configCPU_CLOCK_HZ
/* Ensure the SysTick is clocked at the same frequency as the core. */
#define portNVIC_SYSTICK_CLK_BIT	( 1UL << 2UL )

/* Constants required to manipulate the core.  Registers first... */
#define portNVIC_SYSTICK_CTRL_REG			( * ( ( volatile unsigned long * ) 0xe000e010 ) )
#define portNVIC_SYSTICK_LOAD_REG			( * ( ( volatile unsigned long * ) 0xe000e014 ) )
#define portNVIC_SYSTICK_CURRENT_VALUE_REG	( * ( ( volatile unsigned long * ) 0xe000e018 ) )
#define portNVIC_SYSPRI2_REG				( * ( ( volatile unsigned long * ) 0xe000ed20 ) )
/* ...then bits in the registers. */
#define portNVIC_SYSTICK_INT_BIT			( 1UL << 1UL )
#define portNVIC_SYSTICK_ENABLE_BIT			( 1UL << 0UL )
#define portNVIC_SYSTICK_COUNT_FLAG_BIT		( 1UL << 16UL )
#define portNVIC_PENDSVCLEAR_BIT 			( 1UL << 27UL )
#define portNVIC_PEND_SYSTICK_CLEAR_BIT		( 1UL << 25UL )

/** @brief sets up the system timer to interrupt once every millisecond. Sets Interrupt Priority to highest. 
 *
 *
 *  @return none
 */

void SetupCortexSysTickTimer( void )
{

	/* Configure SysTick to interrupt at the requested rate. */
	portNVIC_SYSTICK_LOAD_REG = ( configSYSTICK_CLOCK_HZ / configTICK_RATE_HZ ) - 1UL;
	portNVIC_SYSTICK_CTRL_REG = ( portNVIC_SYSTICK_CLK_BIT | portNVIC_SYSTICK_INT_BIT | portNVIC_SYSTICK_ENABLE_BIT );
    NVIC_SetPriority (SysTick_IRQn, (1<<__NVIC_PRIO_BITS) - 1);  /* set sys tick timer to highest priority */    
}

/** @brief Sys Tick Interrupt Handler
 *
 *
 *  @return none
 */

void SysTick_Handler(void)
{
    g_MSCtr++; 
}

unsigned long HAL_GetTick(void) {return g_MSCtr;}

/** @brief executes a fibanocci suite for one bus configuration
 *
 *  @param p_Index member of the enumerator terminated with FIBTEST_MAX. Determines which configuration to use.
 *  @param p_Fn An instance of the Fibanoccia function eitehr in internal or external flash
 *  @param p_Implementation member of the enumerator FIBTEST_USE_XXX
 *  @return none
 */

void __attribute((noinline)) FibHelper(unsigned long p_Index,FibonacciFn p_Fn,unsigned long p_Implementation)
{
    unsigned long aLoop;
    for (aLoop = 0; aLoop < MAXFIBONACCITESTS; aLoop++)
        g_FibonacciResults[p_Implementation][p_Index][aLoop][FIBTEST_RESULTITEM_FIBRETURN] = (p_Fn)(aLoop,
                                                                           &g_FibonacciResults[p_Implementation][p_Index][aLoop][FIBTEST_RESULTITEM_TURNAROUND_IN_MS],
                                                                           &g_FibonacciResults[p_Implementation][p_Index][aLoop][FIBTEST_RESULTITEM_DWTCYCLECOUNTER],p_Implementation);  
}

/** @brief executes a full fibanocci suite over all bus configurations
 *
 *  @param p_Implementation member of the enumerator FIBTEST_USE_XXX
 *  @return none
 */

void __attribute((noinline)) FibRunSuite(unsigned long p_Implementation)
{
    FibHelper(FIBTEST_EXTERNALFLASH_INTERNALSRAM,g_FibCallableFn,p_Implementation);
    FibHelper(FIBTEST_INTERNALFLASH_INTERNALSRAM,Fibonacci,p_Implementation);    
    
    
	__asm volatile(
					"	ldr	r3, a_SaveSpLoc		\n" /* Restore the context. */
          " str r13,[r3]          \n"
          " movw r0,#0x1000				\n"
					" movt r0,#0x6400				\n"
          " msr msp, r0         \n"
					"	.align 2						\n"
					"a_SaveSpLoc: .word g_SaveStack				\n"

                 );
    FibHelper(FIBTEST_EXTERNALFLASH_EXTERNALSRAM,g_FibCallableFn,p_Implementation);
    FibHelper(FIBTEST_INTERNALFLASH_EXTERNALSRAM,Fibonacci,p_Implementation);

	__asm volatile(
					"	ldr	r0, a_SaveSpLoc		\n" /* Restore the context. */
          " ldr r0,[r0]          \n"
          " msr msp, r0         \n"
					"	.align 2						\n"
                 );    
    
    __HAL_REMAPMEMORY_FMC();

    FibHelper(FIBTEST_EXTERNALFLASH_MAPPEDTOIBUS_INTERNALSRAM,g_FibCallableFn,p_Implementation);
    FibHelper(FIBTEST_EXTERNALFLASH_MAPPEDTOIBUS_INTERNALSRAM_ZEROBASED,g_FibCallableFn0,p_Implementation);

	__asm volatile(
					"	ldr	r3, a_SaveSpLoc		\n" /* Restore the context. */
          " str r13,[r3]          \n"
          " movw r0,#0x1000				\n"
					" movt r0,#0x6400				\n"
          " msr msp, r0         \n"
                 );    
    
    FibHelper(FIBTEST_EXTERNALFLASH_MAPPEDTOIBUS_EXTERNALSRAM_ZEROBASED,g_FibCallableFn0,p_Implementation);

	__asm volatile(
					"	ldr	r0, a_SaveSpLoc		\n" /* Restore the context. */
          " ldr r0,[r0]          \n"
          " msr msp, r0         \n"
					"	.align 2						\n"
                 );    
        
    __HAL_REMAPMEMORY_FLASH();    

}

/** @brief Application entry point
 *
 *
 *  @return Dummy (required by unused C Runtime Code). Function does not return unless vTaskStartScheduler() has a problem.
 */

int main(void)
{
    unsigned long a_Loop;
    unsigned long a_FibFn = ((unsigned long)&Fibonacci)-0x8000000;
    MX_FMC_Init();

    SetupCortexDWTCycleCounter();
    SetupCortexSysTickTimer();
	__asm volatile(
					" cpsie i				\n" /* Globally enable interrupts. */);

    g_FibCallableFn = (FibonacciFn)(a_FibFn+0x60000000);
    g_FibCallableFn0 = (FibonacciFn)(a_FibFn);
    // This burns a copy of the entire internal flash code into external flash. Note that the
    // Cortex processor already uses PC relative code, so every code that does not read constants
    // from fixed locations is fully reloctable.
    FlashBurnFlash((char *)0x8000000,((unsigned long)(&_sidata))-0x8000000,(char *)0x60000000);
    for (a_Loop = FIBTEST_USE_FULLY_RECURSIVE_IMPLEMENTATION; a_Loop < FIBTEST_USE_MAX; a_Loop++) 
        FibRunSuite(a_Loop);
    while(1);
}
